from dataclasses import dataclass

from model.cell import Cell


@dataclass()
class Board:
    values: dict[int, Cell]

    def __init__(self, values_dict: dict[int, Cell] = None, cells: list[Cell] = None,
                 values_list: list[list[int]] = None):
        if values_dict is not None:
            self.values = values_dict
        else:
            self.values = dict()
            if cells is not None:
                for cell in cells:
                    self.values[cell.id] = cell
            if values_list is not None:
                for i in range(1, len(values_list) + 1):
                    row_values = values_list[i - 1]
                    for j in range(1, len(row_values) + 1):
                        value = row_values[j - 1]
                        cell_id = int(str(i) + str(j))
                        self.values[cell_id] = Cell(cell_id, value)

    @staticmethod
    def load_board(difficulty: str):
        values = list()
        if difficulty.lower() == "easy":
            values.append([2, 0, 5, 0, 0, 7, 4, 8, 0])
            values.append([9, 8, 0, 4, 3, 2, 5, 7, 6])
            values.append([4, 6, 0, 5, 1, 8, 0, 2, 0])

            values.append([0, 9, 0, 6, 0, 4, 8, 1, 2])
            values.append([1, 2, 0, 8, 5, 9, 6, 0, 7])
            values.append([6, 7, 8, 0, 0, 1, 9, 4, 0])

            values.append([3, 1, 6, 0, 8, 0, 0, 9, 4])
            values.append([0, 0, 9, 0, 4, 3, 0, 6, 0])
            values.append([8, 4, 0, 1, 9, 6, 7, 5, 3])
        elif difficulty.lower() == "medium":
            values.append([0, 0, 0, 0, 6, 8, 2, 7, 3])
            values.append([3, 0, 2, 0, 0, 5, 8, 6, 0])
            values.append([1, 8, 6, 0, 7, 3, 0, 0, 9])

            values.append([5, 6, 7, 3, 0, 0, 0, 9, 0])
            values.append([9, 0, 4, 0, 0, 6, 3, 8, 7])
            values.append([8, 0, 0, 9, 1, 7, 0, 5, 0])

            values.append([2, 5, 9, 0, 4, 1, 0, 3, 6])
            values.append([6, 0, 0, 7, 3, 0, 0, 0, 5])
            values.append([0, 3, 1, 0, 0, 9, 4, 2, 8])
        elif difficulty.lower() == "hard":
            values.append([0, 5, 0, 1, 2, 0, 0, 0, 6])
            values.append([8, 1, 7, 0, 3, 0, 5, 4, 0])
            values.append([2, 0, 0, 5, 8, 0, 0, 1, 0])

            values.append([9, 8, 4, 0, 0, 0, 2, 5, 1])
            values.append([0, 3, 0, 8, 0, 0, 7, 0, 0])
            values.append([0, 2, 0, 4, 9, 5, 6, 0, 8])

            values.append([0, 0, 9, 7, 5, 8, 4, 0, 0])
            values.append([5, 7, 0, 2, 4, 0, 0, 6, 9])
            values.append([3, 0, 2, 9, 0, 1, 8, 7, 0])
        else:
            for i in range(0, 9):
                values.append([0, 0, 0, 0, 0, 0, 0, 0, 0])
        return Board(values_list=values)

    @staticmethod
    def __get_row_constraints():
        row_constraints = list()
        row_constraints.append({11, 12, 13, 14, 15, 16, 17, 18, 19})
        row_constraints.append({21, 22, 23, 24, 25, 26, 27, 28, 29})
        row_constraints.append({31, 32, 33, 34, 35, 36, 37, 38, 39})
        row_constraints.append({41, 42, 43, 44, 45, 46, 47, 48, 49})
        row_constraints.append({51, 52, 53, 54, 55, 56, 57, 58, 59})
        row_constraints.append({61, 62, 63, 64, 65, 66, 67, 68, 69})
        row_constraints.append({71, 72, 73, 74, 75, 76, 77, 78, 79})
        row_constraints.append({81, 82, 83, 84, 85, 86, 87, 88, 89})
        row_constraints.append({91, 92, 93, 94, 95, 96, 97, 98, 99})
        return row_constraints

    @staticmethod
    def __get_column_constraints():
        column_constraints = list()
        column_constraints.append({11, 21, 31, 41, 51, 61, 71, 81, 91})
        column_constraints.append({12, 22, 32, 42, 52, 62, 72, 82, 92})
        column_constraints.append({13, 23, 33, 43, 53, 63, 73, 83, 93})
        column_constraints.append({14, 24, 34, 44, 54, 64, 74, 84, 94})
        column_constraints.append({15, 25, 35, 45, 55, 65, 75, 85, 95})
        column_constraints.append({16, 26, 36, 46, 56, 66, 76, 86, 96})
        column_constraints.append({17, 27, 37, 47, 57, 67, 77, 87, 97})
        column_constraints.append({18, 28, 38, 48, 58, 68, 78, 88, 98})
        column_constraints.append({19, 29, 39, 49, 59, 69, 79, 89, 99})
        return column_constraints

    @staticmethod
    def __get_square_constraints():
        square_constraints = list()
        square_constraints.append({11, 12, 13, 21, 22, 23, 31, 32, 33})
        square_constraints.append({14, 15, 16, 24, 25, 26, 34, 35, 36})
        square_constraints.append({17, 18, 19, 27, 28, 29, 37, 38, 39})
        square_constraints.append({41, 42, 43, 51, 52, 53, 61, 62, 63})
        square_constraints.append({44, 45, 46, 54, 55, 56, 64, 65, 66})
        square_constraints.append({47, 48, 49, 57, 58, 59, 67, 68, 69})
        square_constraints.append({71, 72, 73, 81, 82, 83, 91, 92, 93})
        square_constraints.append({74, 75, 76, 84, 85, 86, 94, 95, 96})
        square_constraints.append({77, 78, 79, 87, 88, 89, 97, 98, 99})
        return square_constraints

    @staticmethod
    def __check_for_different_values(cells: list[Cell]):
        cell_values = map(lambda cell: cell.value, cells)
        set_cell_values = filter(lambda cell_value: cell_value > 0, cell_values)
        return len(set(set_cell_values)) == 9

    def verify(self):
        for i in range(1, 4):
            if i == 1:
                current_constraints = Board.__get_row_constraints()
                constraint_type = "row"
            elif i == 2:
                current_constraints = Board.__get_column_constraints()
                constraint_type = "column"
            else:
                current_constraints = Board.__get_square_constraints()
                constraint_type = "square"
            for j in range(1, len(current_constraints) + 1):
                current_constraint = current_constraints[j - 1]
                cells_in_constraint = filter(lambda cell: cell.id in current_constraint, self.values.values())
                if not Board.__check_for_different_values(list(cells_in_constraint)):
                    return "The values in " + constraint_type + " " + str(j) + " are not unique."
        return "Congratulations! You have solved the Sudoku."
