import uvicorn
from fastapi import FastAPI
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import RedirectResponse

from model.board import Board
from model.cell import Cell
from model.difficulty import difficulties, Difficulty

app = FastAPI()

origins = [
    "http://localhost:4200",
    "http://localhost:3000",
    "http://localhost:8080"
]

app.add_middleware(
    CORSMiddleware,
    allow_origins=origins,
    allow_methods=["*"],
    allow_headers=["*"]
)


@app.get("/", response_class=RedirectResponse)
async def root():
    return "/difficulties"


@app.get("/difficulties", response_model=list[Difficulty])
async def get_difficulties():
    return difficulties


@app.get("/board", response_model=Board)
async def get_board(difficulty: str = ""):
    return Board.load_board(difficulty)


@app.post("/verify", response_model=dict[str, str])
async def verify(cells: list[Cell]):
    values_map = dict()
    for cell in cells:
        values_map[cell.id] = cell
    board = Board(values=values_map)
    return {"result": board.verify()}


if __name__ == "__main__":
    uvicorn.run("sudoku:app", host="localhost", port=8081, reload=True)
