import React from "react";
import {Button, Container, Table} from "react-bootstrap";
import {backendURL, CheckError, withRouter} from "../constants";
import Board from "../model/Board";
import SquareComponent from "./SquareComponent";

class BoardComponent extends React.Component {

    constructor(props) {
        super(props);
        this.state = {
            difficulty: "",
            board: undefined,
            verificationRunning: false
        };
    }

    componentDidMount() {
        const difficulty = this.props.router.params.difficulty;
        this.setState({
            difficulty: difficulty
        });
        fetch(backendURL + "/board?difficulty=" + difficulty)
            .then(CheckError)
            .then((jsonResponse) => {
                this.setState({
                    board: new Board(jsonResponse.values)
                });
            })
            .catch((error) => {
                console.error(error);
                window.alert(error.message);
            });
    }

    verify() {
        this.setState({
            verificationRunning: true
        });

        const numMissingValues = this.state.board.countMissingValues();
        if (numMissingValues === 0) {
            const options = {
                method: 'POST',
                body: JSON.stringify(Array.from(this.state.board.values.values())),
                headers: {
                    'Content-Type': 'application/json'
                }
            };

            fetch(backendURL + "/verify", options)
                .then(CheckError)
                .then((jsonResponse) => {
                    const message = jsonResponse.result;
                    window.alert(message);
                    this.setState({
                        verificationRunning: false
                    });
                })
                .catch((error) => {
                    console.error(error);
                    window.alert(error.message);
                    this.setState({
                        verificationRunning: false
                    });
                });
        } else {
            let message = " fields are currently empty).";
            if (numMissingValues === 1) {
                message = " field is currently empty).";
            }
            message = "Please enter a value for every field (" + numMissingValues + message;
            window.alert(message);
            this.setState({
                verificationRunning: false
            });
        }
    }

    render() {
        let board;
        if (this.state.board === undefined) {
            board = <h5 className="text-center">Loading the board...</h5>;
        } else {
            board = <Table>
                <tbody>
                {[1, 2, 3].map((row) =>
                    <tr className="d-flex" key={'boardRow' + row}>
                        {[1, 2, 3].map((column) =>
                            <td className="border-0 col-4" key={'boardColumn' + column + 'Row' + row}>
                                <SquareComponent board={this.state.board} outerColumn={column} outerRow={row}/>
                            </td>
                        )}
                    </tr>
                )}
                </tbody>
            </Table>;
        }

        let verifyButton = null;
        if (this.state.board !== undefined && !this.state.verificationRunning) {
            verifyButton = <Button variant="secondary" onClick={() => this.verify()}>
                Verify
            </Button>;
        } else if (this.state.verificationRunning) {
            verifyButton = <Button variant="secondary" disabled>
                Verifying...
            </Button>
        }

        return (
            <div className="Board">
                <Container fluid>
                    <h5 className="text-center">Difficulty: {this.state.difficulty}</h5>

                    {board}

                    <div className="text-center">
                        {verifyButton}
                    </div>
                </Container>
            </div>
        )
    }
}

export default withRouter(BoardComponent);