package de.uniwue.swpexample.springbackend.model;

import java.util.*;
import java.util.stream.Collectors;

public class Board {

  private final Map<Integer, Cell> values;

  public Board(Cell[] cells) {
    this.values = new HashMap<>();
    for (Cell cell : cells) {
      values.put(cell.getId(), cell);
    }
  }

  public Board(List<List<Integer>> values) {
    this.values = new HashMap<>();
    for (int i = 1; i <= values.size(); i++) {
      List<Integer> rowValues = values.get(i - 1);
      for (int j = 1; j <= rowValues.size(); j++) {
        int value = rowValues.get(j - 1);
        int cellId = Integer.parseInt("" + i + j);
        this.values.put(cellId, new Cell(cellId, value));
      }
    }
  }

  public static Board loadBoard(String difficulty) {
    List<List<Integer>> values = new ArrayList<>();
    if (difficulty.equalsIgnoreCase("easy")) {
      values.add(List.of(2, 0, 5, 0, 0, 7, 4, 8, 0));
      values.add(List.of(9, 8, 0, 4, 3, 2, 5, 7, 6));
      values.add(List.of(4, 6, 0, 5, 1, 8, 0, 2, 0));

      values.add(List.of(0, 9, 0, 6, 0, 4, 8, 1, 2));
      values.add(List.of(1, 2, 0, 8, 5, 9, 6, 0, 7));
      values.add(List.of(6, 7, 8, 0, 0, 1, 9, 4, 0));

      values.add(List.of(3, 1, 6, 0, 8, 0, 0, 9, 4));
      values.add(List.of(0, 0, 9, 0, 4, 3, 0, 6, 0));
      values.add(List.of(8, 4, 0, 1, 9, 6, 7, 5, 3));
    } else if (difficulty.equalsIgnoreCase("medium")) {
      values.add(List.of(0, 0, 0, 0, 6, 8, 2, 7, 3));
      values.add(List.of(3, 0, 2, 0, 0, 5, 8, 6, 0));
      values.add(List.of(1, 8, 6, 0, 7, 3, 0, 0, 9));

      values.add(List.of(5, 6, 7, 3, 0, 0, 0, 9, 0));
      values.add(List.of(9, 0, 4, 0, 0, 6, 3, 8, 7));
      values.add(List.of(8, 0, 0, 9, 1, 7, 0, 5, 0));

      values.add(List.of(2, 5, 9, 0, 4, 1, 0, 3, 6));
      values.add(List.of(6, 0, 0, 7, 3, 0, 0, 0, 5));
      values.add(List.of(0, 3, 1, 0, 0, 9, 4, 2, 8));
    } else if (difficulty.equalsIgnoreCase("hard")) {
      values.add(List.of(0, 5, 0, 1, 2, 0, 0, 0, 6));
      values.add(List.of(8, 1, 7, 0, 3, 0, 5, 4, 0));
      values.add(List.of(2, 0, 0, 5, 8, 0, 0, 1, 0));

      values.add(List.of(9, 8, 4, 0, 0, 0, 2, 5, 1));
      values.add(List.of(0, 3, 0, 8, 0, 0, 7, 0, 0));
      values.add(List.of(0, 2, 0, 4, 9, 5, 6, 0, 8));

      values.add(List.of(0, 0, 9, 7, 5, 8, 4, 0, 0));
      values.add(List.of(5, 7, 0, 2, 4, 0, 0, 6, 9));
      values.add(List.of(3, 0, 2, 9, 0, 1, 8, 7, 0));
    } else {
      for (int i = 1; i <= 9; i++) {
        values.add(List.of(0, 0, 0, 0, 0, 0, 0, 0, 0));
      }
    }
    return new Board(values);
  }

  private static List<Set<Integer>> getRowConstraints() {
    List<Set<Integer>> rowConstraints = new ArrayList<>();
    rowConstraints.add(Set.of(11, 12, 13, 14, 15, 16, 17, 18, 19));
    rowConstraints.add(Set.of(21, 22, 23, 24, 25, 26, 27, 28, 29));
    rowConstraints.add(Set.of(31, 32, 33, 34, 35, 36, 37, 38, 39));
    rowConstraints.add(Set.of(41, 42, 43, 44, 45, 46, 47, 48, 49));
    rowConstraints.add(Set.of(51, 52, 53, 54, 55, 56, 57, 58, 59));
    rowConstraints.add(Set.of(61, 62, 63, 64, 65, 66, 67, 68, 69));
    rowConstraints.add(Set.of(71, 72, 73, 74, 75, 76, 77, 78, 79));
    rowConstraints.add(Set.of(81, 82, 83, 84, 85, 86, 87, 88, 89));
    rowConstraints.add(Set.of(91, 92, 93, 94, 95, 96, 97, 98, 99));
    return rowConstraints;
  }

  private static List<Set<Integer>> getColumnConstraints() {
    List<Set<Integer>> columnConstraints = new ArrayList<>();
    columnConstraints.add(Set.of(11, 21, 31, 41, 51, 61, 71, 81, 91));
    columnConstraints.add(Set.of(12, 22, 32, 42, 52, 62, 72, 82, 92));
    columnConstraints.add(Set.of(13, 23, 33, 43, 53, 63, 73, 83, 93));
    columnConstraints.add(Set.of(14, 24, 34, 44, 54, 64, 74, 84, 94));
    columnConstraints.add(Set.of(15, 25, 35, 45, 55, 65, 75, 85, 95));
    columnConstraints.add(Set.of(16, 26, 36, 46, 56, 66, 76, 86, 96));
    columnConstraints.add(Set.of(17, 27, 37, 47, 57, 67, 77, 87, 97));
    columnConstraints.add(Set.of(18, 28, 38, 48, 58, 68, 78, 88, 98));
    columnConstraints.add(Set.of(19, 29, 39, 49, 59, 69, 79, 89, 99));
    return columnConstraints;
  }

  private static List<Set<Integer>> getSquareConstraints() {
    List<Set<Integer>> squareConstraints = new ArrayList<>();
    squareConstraints.add(Set.of(11, 12, 13, 21, 22, 23, 31, 32, 33));
    squareConstraints.add(Set.of(14, 15, 16, 24, 25, 26, 34, 35, 36));
    squareConstraints.add(Set.of(17, 18, 19, 27, 28, 29, 37, 38, 39));
    squareConstraints.add(Set.of(41, 42, 43, 51, 52, 53, 61, 62, 63));
    squareConstraints.add(Set.of(44, 45, 46, 54, 55, 56, 64, 65, 66));
    squareConstraints.add(Set.of(47, 48, 49, 57, 58, 59, 67, 68, 69));
    squareConstraints.add(Set.of(71, 72, 73, 81, 82, 83, 91, 92, 93));
    squareConstraints.add(Set.of(74, 75, 76, 84, 85, 86, 94, 95, 96));
    squareConstraints.add(Set.of(77, 78, 79, 87, 88, 89, 97, 98, 99));
    return squareConstraints;
  }

  private static boolean checkForDifferentValues(List<Cell> cells) {
    return cells.stream().map(Cell::getValue).filter(value -> value > 0).collect(Collectors.toSet()).size() == 9;
  }

  public Map<Integer, Cell> getValues() {
    return values;
  }

  public String verify() {
    for (int i = 1; i <= 3; i++) {
      List<Set<Integer>> currentConstraints;
      String constraintType;
      if (i == 1) {
        currentConstraints = getRowConstraints();
        constraintType = "row";
      } else if (i == 2) {
        currentConstraints = getColumnConstraints();
        constraintType = "column";
      } else {
        currentConstraints = getSquareConstraints();
        constraintType = "square";
      }
      for (int j = 1; j <= currentConstraints.size(); j++) {
        Set<Integer> currentConstraint = currentConstraints.get(j - 1);
        List<Cell> cellsInConstraint = this.values.values().stream()
                .filter(cell -> currentConstraint.contains(cell.getId())).collect(Collectors.toList());
        if (!checkForDifferentValues(cellsInConstraint)) {
          return "The values in " + constraintType + " " + j + " are not unique.";
        }
      }
    }
    return "Congratulations! You have solved the Sudoku.";
  }
}
