import {Injectable} from '@angular/core';
import {Difficulty} from "./model/difficulty";
import {Board, VerificationResult} from "./model/board";
import {HttpClient, HttpErrorResponse} from "@angular/common/http";
import {catchError} from "rxjs/operators";
import {Observable, of, throwError} from "rxjs";

@Injectable({
  providedIn: 'root'
})
export class SudokuService {

  private backendUrl = "http://localhost:8081";

  constructor(
    private http: HttpClient
  ) {
  }

  private static handleError(error: HttpErrorResponse) {
    let errorMessage = "Error: " + error.message;
    console.error(error);
    window.alert(errorMessage);
    return throwError(() => errorMessage);
  }

  getDifficulties(): Observable<Difficulty[]> {
    return this.http.get<Difficulty[]>(this.backendUrl + "/difficulties")
      .pipe(
        catchError(SudokuService.handleError)
      );
  }

  getBoard(difficulty: String): Observable<Board> {
    return this.http.get<Board>(this.backendUrl + "/board?difficulty=" + difficulty)
      .pipe(
        catchError(SudokuService.handleError)
      );
  }

  verifyBoard(board: Board): Observable<VerificationResult> {
    let numMissingValues = board.countMissingValues();
    if (numMissingValues === 0) {
      return this.http.post<VerificationResult>(this.backendUrl + "/verify", Array.from(board.values.values()))
        .pipe(
          catchError(SudokuService.handleError)
        );
    } else {
      let appendix = " fields are currently empty).";
      if (numMissingValues === 1) {
        appendix = " field is currently empty).";
      }
      return of(new VerificationResult("Please enter a value for every field (" + numMissingValues + appendix));
    }
  }

}
